"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return DomainClient;
    }
});
let DomainClient = class DomainClient {
    /**
   * Returns domain client domain
   * @returns {String} client domain
   */ get domain() {
        return this._domain;
    }
    /**
   * Returns domain client token
   * @returns {String} client token
   */ get token() {
        return this._token;
    }
    /**
   * Sends an authorized json API request
   * @param {Object} opts options request options
   * @param {Boolean} [isExtendedTimeout] whether to run the request with an extended timeout
   * @returns {Promise<Object|String|any>} request result
   */ async requestApi(opts, isExtendedTimeout = false) {
        await this._updateHost();
        try {
            return await this._httpClient.request(Object.assign({}, opts, {
                headers: opts.headers || {
                    "auth-token": this._token
                },
                url: this._urlCache.url + opts.url,
                json: true
            }), isExtendedTimeout);
        } catch (err) {
            if (![
                "ConflictError",
                "InternalError",
                "ApiError",
                "TimeoutError"
            ].includes(err.name)) {
                throw err;
            } else {
                if (this._regionCache.length === this._regionIndex + 1) {
                    this._regionIndex = 0;
                    throw err;
                } else {
                    this._regionIndex++;
                    return this.requestApi(opts);
                }
            }
        }
    }
    /**
   * Sends an http request
   * @param {Object} opts options request options
   * @returns {Promise<Object|String|any>} request result
   */ request(opts) {
        return this._httpClient.request(opts);
    }
    async _updateHost() {
        if (!this._urlCache || this._urlCache.lastUpdated < Date.now() - 1000 * 60 * 10) {
            await this._updateRegions();
            const urlSettings = await this._httpClient.request({
                url: `https://mt-provisioning-api-v1.${this._domain}/users/current/servers/mt-client-api`,
                method: "GET",
                headers: {
                    "auth-token": this._token
                },
                json: true
            });
            this._urlCache = {
                url: `https://${this._apiPath}.${this._regionCache[this._regionIndex]}.${urlSettings.domain}`,
                domain: urlSettings.domain,
                lastUpdated: Date.now()
            };
        } else {
            this._urlCache = {
                url: `https://${this._apiPath}.${this._regionCache[this._regionIndex]}.${this._urlCache.domain}`,
                domain: this._urlCache.domain,
                lastUpdated: Date.now()
            };
        }
    }
    async _updateRegions() {
        this._regionIndex = 0;
        this._regionCache = await this._httpClient.request({
            url: `https://mt-provisioning-api-v1.${this._domain}/users/current/regions`,
            method: "GET",
            headers: {
                "auth-token": this._token
            },
            json: true
        });
    }
    /**
   * Constructs domain client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} apiPath api url part
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */ constructor(httpClient, token, apiPath, domain = "agiliumtrade.agiliumtrade.ai"){
        this._httpClient = httpClient;
        this._apiPath = apiPath;
        this._domain = domain;
        this._token = token;
        this._urlCache = null;
        this._regionCache = [];
        this._regionIndex = 0;
    }
};

//# sourceMappingURL=data:application/json;base64,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