'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import LoggerManager from '../logger';
import MetaApiConnection from './metaApiConnection';
import TimeoutError from '../clients/timeoutError';
let RpcMetaApiConnection = class RpcMetaApiConnection extends MetaApiConnection {
    /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @param {string} instanceId connection instance id
   * @return {Promise} promise resolving when the connection is opened
   */ connect(instanceId) {
        var _this = this;
        return _async_to_generator(function*() {
            if (!_this._openedInstances.includes(instanceId)) {
                _this._openedInstances.push(instanceId);
            }
            if (!_this._opened) {
                _this._opened = true;
                const accountRegions = _this._account.accountRegions;
                _this._websocketClient.addAccountCache(_this._account.id, accountRegions);
                Object.keys(accountRegions).forEach((region)=>{
                    if (!_this._options.region || _this._options.region === region) {
                        _this._websocketClient.ensureSubscribe(accountRegions[region], 0);
                        _this._websocketClient.ensureSubscribe(accountRegions[region], 1);
                    }
                });
            }
        })();
    }
    /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   * @param {string} instanceId connection instance id
   */ close(instanceId) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._opened) {
                _this._openedInstances = _this._openedInstances.filter((id)=>id !== instanceId);
                if (!_this._openedInstances.length && !_this._closed) {
                    clearInterval(_this._refreshJob);
                    yield _this._connectionRegistry.removeRpc(_this.account);
                    _this._websocketClient.removeSynchronizationListener(_this.account.id, _this);
                    _this._websocketClient.removeAccountCache(_this.account.id);
                    _this._websocketClient.removeReconnectListener(_this);
                    _this._closed = true;
                }
            }
        })();
    }
    /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   * @param {Number} replicas number of account replicas launched
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onConnected(instanceIndex, replicas) {
        var _this = this;
        return _async_to_generator(function*() {
            const state = _this._getState(instanceIndex);
            state.synchronized = true;
            const region = _this.getRegion(instanceIndex);
            _this.cancelRefresh(region);
        })();
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onDisconnected(instanceIndex) {
        var _this = this;
        return _async_to_generator(function*() {
            const state = _this._getState(instanceIndex);
            state.synchronized = false;
            _this._logger.debug(`${_this._account.id}:${instanceIndex}: disconnected from broker`);
        })();
    }
    /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   */ onStreamClosed(instanceIndex) {
        var _this = this;
        return _async_to_generator(function*() {
            delete _this._stateByInstanceIndex[instanceIndex];
        })();
    }
    /**
   * Returns flag indicating status of state synchronization with MetaTrader terminal
   * @returns {Boolean} a flag indicating status of state synchronization with MetaTrader terminal
   */ isSynchronized() {
        return Object.values(this._stateByInstanceIndex).map((instance)=>instance.synchronized).includes(true);
    }
    /**
   * Waits until synchronization to RPC application is completed
   * @param {Number} timeoutInSeconds synchronization timeout in seconds. Defaults to 5 minutes
   * @return {Promise} promise which resolves when synchronization to RPC application is completed
   * @throws {TimeoutError} if application failed to synchronize with the teminal within timeout allowed
   */ waitSynchronized(timeoutInSeconds = 300) {
        var _this = this;
        return _async_to_generator(function*() {
            _this._checkIsConnectionActive();
            const startTime = Date.now();
            let synchronized = _this.isSynchronized();
            while(!synchronized && startTime + timeoutInSeconds * 1000 > Date.now()){
                yield new Promise((res)=>setTimeout(res, 1000));
                synchronized = _this.isSynchronized();
            }
            if (!synchronized) {
                throw new TimeoutError('Timed out waiting for MetaApi to synchronize to MetaTrader account ' + _this._account.id);
            }
            // eslint-disable-next-line
            while(true){
                try {
                    yield _this._websocketClient.waitSynchronized(_this._account.id, undefined, 'RPC', 5, 'RPC');
                    break;
                } catch (err) {
                    if (Date.now() > startTime + timeoutInSeconds * 1000) {
                        throw err;
                    }
                }
            }
        })();
    }
    /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect
   * @param {String} region reconnected region
   * @param {Number} instanceNumber reconnected instance number
   * @return {Promise} promise which resolves when connection to MetaApi websocket API restored after a disconnect
   */ onReconnected(region, instanceNumber) {
        var _this = this;
        return _async_to_generator(function*() {
            const instanceTemplate = `${region}:${instanceNumber}`;
            Object.keys(_this._stateByInstanceIndex).filter((key)=>key.startsWith(`${instanceTemplate}:`)).forEach((key)=>{
                delete _this._stateByInstanceIndex[key];
            });
        })();
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex[instanceIndex]) {
            this._stateByInstanceIndex[instanceIndex] = {
                instanceIndex,
                synchronized: false
            };
        }
        return this._stateByInstanceIndex[instanceIndex];
    }
    /**
   * Constructs MetaApi MetaTrader RPC Api connection
   * @param {MetaApiOpts} options MetaApi options
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   */ constructor(options, websocketClient, account, connectionRegistry){
        super(options, websocketClient, account, 'RPC');
        this._connectionRegistry = connectionRegistry;
        this._websocketClient.addSynchronizationListener(account.id, this);
        this._stateByInstanceIndex = {};
        this._openedInstances = [];
        Object.values(account.accountRegions).forEach((replicaId)=>this._websocketClient.addReconnectListener(this, replicaId));
        this._logger = LoggerManager.getLogger('MetaApiConnection');
    }
};
/**
 * Exposes MetaApi MetaTrader RPC API connection to consumers
 */ export { RpcMetaApiConnection as default };

//# sourceMappingURL=data:application/json;base64,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